<?php
session_start();
$ds = DIRECTORY_SEPARATOR;
$project_root_raw = dirname(dirname(__DIR__)); 
$project_root_1 = rtrim($project_root_raw, '/\\'); 

$db_path = $project_root_1 . $ds . 'includes' . $ds . 'db_connection.php';
$functions_path = $project_root_1 . $ds . 'includes' . $ds . 'functions.php';

if (file_exists($db_path) && file_exists($functions_path)) {
    include $db_path;
    include $functions_path;
} else {
    // এই ফাইলগুলো পাওয়া না গেলে মারাত্মক ত্রুটি দেখাবে।
    die("FATAL ERROR: Required core files not found.");
}

// লগইন ও অনুমতি চেক
if (!is_logged_in() || (isset($_SESSION['role']) && $_SESSION['role'] !== 'Admin')) {
    redirect_to('../../index.php'); 
}

// ইনপুট ডেটা সংগ্রহ ও বৈধতা যাচাই
$student_id = filter_input(INPUT_GET, 'student_id', FILTER_VALIDATE_INT);
$exam_id = filter_input(INPUT_GET, 'exam_id', FILTER_VALIDATE_INT);
$class_id = filter_input(INPUT_GET, 'class_id', FILTER_VALIDATE_INT); 

$student_data = null;
$exam_name = '';
$results = []; // বিষয়ভিত্তিক ফলাফলের অ্যারে
$total_marks = 0;
$error_message = null;

if (!$student_id || !$exam_id || !$class_id) {
    $error_message = "অসম্পূর্ণ ডেটা: শিক্ষার্থী, পরীক্ষা বা ক্লাস আইডি অনুপস্থিত।";
}

if (!isset($error_message) && $conn) {
    
    // --- ১. শিক্ষার্থীর তথ্য লোড করা ---
    $sql_student = "
        SELECT 
            s.student_id,
            COALESCE(CONCAT(s.first_name, ' ', s.last_name), 'নাম নেই') AS full_name, 
            s.class_roll,
            c.class_name
        FROM 
            students s
        JOIN 
            classes c ON s.class_id = c.class_id
        WHERE 
            s.student_id = ? AND s.class_id = ?
    ";
    
    if ($stmt = $conn->prepare($sql_student)) {
        $stmt->bind_param("ii", $student_id, $class_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $student_data = $result->fetch_assoc();
        }
        $stmt->close();
    } else {
        $error_message = "শিক্ষার্থী তথ্য ক্যোয়ারি ত্রুটি: " . $conn->error;
        error_log("SQL Error (Student Info): " . $conn->error);
    }
    
    // যদি শিক্ষার্থীর তথ্য না পাওয়া যায়
    if (!$student_data) {
        $error_message = "শিক্ষার্থীর তথ্য খুঁজে পাওয়া যায়নি। আইডি সঠিক কিনা নিশ্চিত করুন।";
    }

    // --- ২. পরীক্ষার নাম লোড করা ---
    if (!$error_message) {
        $sql_exam = "SELECT exam_name FROM exams WHERE exam_id = ?";
        if ($stmt = $conn->prepare($sql_exam)) {
            $stmt->bind_param("i", $exam_id);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($row = $result->fetch_assoc()) {
                $exam_name = $row['exam_name'];
            }
            $stmt->close();
        } else {
            error_log("Exam Query Error: " . $conn->error);
        }

        // --- ৩. ফলাফল লোড করা (exam_results টেবিল ও marks কলাম ব্যবহার করা হলো) ---
        $sql_results = "
            SELECT 
                r.marks,  
                s.subject_name
            FROM 
                exam_results r  
            JOIN 
                subjects s ON r.subject_id = s.subject_id
            WHERE 
                r.student_id = ? AND r.exam_id = ? 
            ORDER BY 
                s.subject_name ASC
        ";
        
        if ($stmt = $conn->prepare($sql_results)) {
            $stmt->bind_param("ii", $student_id, $exam_id); 
            
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                while ($row = $result->fetch_assoc()) {
                    // 'marks' কলাম থেকে ডেটা লোড করে 'marks_obtained' কী-তে রাখা হলো
                    $results[] = [
                        'subject_name' => $row['subject_name'],
                        'marks_obtained' => $row['marks'] 
                    ];
                    $total_marks += (int)$row['marks']; 
                }
            } else {
                $error_message = "ফলাফল ক্যোয়ারি এক্সিকিউশন ত্রুটি: " . $stmt->error;
                error_log("SQL Error (Results): " . $stmt->error);
            }
            $stmt->close();
        } else {
            $error_message = "ফলাফল ক্যোয়ারি প্রস্তুতি ত্রুটি: " . $conn->error;
            error_log("SQL Error (Prepare Results): " . $conn->error);
        }
        
        // যদি ফলাফল না পাওয়া যায়
        if (empty($results)) {
             $error_message = "এই পরীক্ষার জন্য শিক্ষার্থীর কোনো ফলাফল জমা দেওয়া হয়নি।";
        }
    }
    
    $conn->close();
}

// গ্রেড গণনা ফাংশন
if (!function_exists('calculate_grade')) {
    function calculate_grade($mark) {
        if ($mark >= 80) return 'A+';
        if ($mark >= 70) return 'A';
        if ($mark >= 60) return 'A-';
        if ($mark >= 50) return 'B';
        if ($mark >= 40) return 'C';
        if ($mark >= 33) return 'D';
        return 'F';
    }
}
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>মার্কশিট - <?php echo htmlspecialchars($exam_name); ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        /* মূল স্টাইল */
        body { background-color: #f0f2f5; font-family: 'Arial', sans-serif; display: flex; justify-content: center; align-items: flex-start; padding: 20px; min-height: 100vh; }
        .container { display: flex; flex-direction: column; align-items: center; width: 100%; max-width: 800px; }
        .mark-sheet {
            width: 100%;
            background-color: #ffffff;
            border: 2px solid #3498db;
            border-radius: 15px;
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.15);
            padding: 30px;
        }
        .header {
            text-align: center;
            border-bottom: 3px solid #3498db;
            padding-bottom: 15px;
            margin-bottom: 20px;
        }
        .header h1 {
            color: #2c3e50;
            margin: 0;
            font-size: 1.8em;
        }
        .header h2 {
            color: #3498db;
            margin: 5px 0 10px 0;
            font-size: 1.2em;
        }
        .student-info {
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
            margin-bottom: 25px;
            padding: 10px;
            background-color: #ecf0f1;
            border-radius: 8px;
        }
        .student-info div {
            flex: 1 1 45%;
            margin: 5px 0;
            font-size: 1.05em;
            color: #34495e;
        }
        .student-info strong {
            color: #2c3e50;
            margin-right: 5px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }
        table th, table td {
            border: 1px solid #bdc3c7;
            padding: 12px 15px;
            text-align: center;
        }
        table th {
            background-color: #3498db;
            color: white;
            font-weight: bold;
            text-transform: uppercase;
        }
        table tr:nth-child(even) {
            background-color: #f7f9fb;
        }
        .summary {
            text-align: right;
            font-size: 1.1em;
            font-weight: bold;
            padding-top: 15px;
            border-top: 2px solid #3498db;
            color: #2c3e50;
            margin-bottom: 20px;
        }
        .error-box {
            background-color: #fce4e4;
            color: #c0392b;
            border: 1px solid #c0392b;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
            font-size: 1.2em;
            margin-top: 50px;
        }

        /* নতুন যোগ করা CSS: বাটন স্টাইল */
        .action-buttons {
            display: flex;
            justify-content: space-between;
            width: 100%;
            max-width: 800px;
            margin-bottom: 20px;
            gap: 10px; /* বাটনগুলোর মধ্যে ফাঁকা স্থান */
        }
        .btn {
            padding: 10px 15px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 1em;
            font-weight: bold;
            transition: background-color 0.3s, transform 0.1s;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            flex-grow: 1; /* সকল বাটনকে সমান প্রস্থ দেওয়া */
            justify-content: center;
            white-space: nowrap;
        }
        .btn-print {
            background-color: #2ecc71;
            color: white;
        }
        .btn-print:hover {
            background-color: #27ae60;
            transform: translateY(-1px);
        }
        .btn-back, .btn-secondary {
            background-color: #e74c3c;
            color: white;
        }
        .btn-back:hover, .btn-secondary:hover {
            background-color: #c0392b;
            transform: translateY(-1px);
        }
        
        /* মন্তব্য এবং স্বাক্ষর বিভাগ */
        .comments-and-signature {
            display: flex;
            justify-content: space-between;
            margin-top: 40px;
            padding-top: 20px;
            border-top: 1px solid #bdc3c7;
            flex-wrap: wrap;
        }
        .comment-box {
            width: 48%; 
            margin-bottom: 20px;
            border: 1px solid #bdc3c7;
            border-radius: 5px;
            padding: 10px;
        }
        .signature-box {
            width: 100%;
            text-align: right;
            margin-top: 20px;
        }
        .signature-line {
            display: inline-block;
            border-top: 1px solid #2c3e50;
            padding: 5px 0 0;
            margin-top: 40px;
            width: 250px;
            text-align: center;
            font-weight: bold;
        }

        /* প্রিন্ট স্টাইল */
        @media print {
            body { 
                background-color: white !important; 
                padding: 0;
            }
            .container {
                max-width: 100%;
            }
            .mark-sheet {
                border: none;
                box-shadow: none;
                border-radius: 0;
                padding: 0;
            }
            .action-buttons {
                display: none; 
            }
            .comment-box {
                border: 1px solid #333;
            }
            table th, table td {
                border: 1px solid #333;
            }
            .signature-line {
                border-top: 1px solid #333;
            }
            .student-info {
                background-color: #f7f9fb; 
                -webkit-print-color-adjust: exact;
            }
        }
        /* মোবাইল রেসপনসিভনেস */
        @media (max-width: 650px) {
             .action-buttons {
                flex-direction: column;
            }
            .btn {
                 flex-grow: 1;
            }
            .student-info div {
                flex: 1 1 100%;
            }
            .comment-box {
                width: 100%;
            }
        }
    </style>
</head>
<body>

<div class="container">
    
    <?php if (!isset($error_message) && $student_data): ?>
        <div class="action-buttons">
            <a href="../../public/dashboard.php" class="btn btn-secondary" style="background-color: #555;">
                <i class="fas fa-home"></i> হোম
            </a>
            <button onclick="window.print()" class="btn btn-print">
                <i class="fas fa-print"></i> প্রিন্ট মার্কশিট
            </button>
            <a href="mark_sheet_search.php" class="btn btn-secondary" style="background-color: #f39c12;">
                <i class="fas fa-search"></i> নতুন অনুসন্ধান
            </a>
            <a href="comparative_mark_sheet.php?student_id=<?php echo $student_id; ?>&class_id=<?php echo $class_id; ?>" class="btn btn-secondary" style="background-color: #3498db;">
                <i class="fas fa-list-alt"></i> তুলনামূলক ফলাফল
            </a>
        </div>
    <?php endif; ?>

    <div class="mark-sheet">
        
        <?php if (isset($error_message)): ?>
            <div class="error-box">
                <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error_message); ?>
            </div>
            <?php if (!empty($student_id) || !empty($exam_id) || !empty($class_id)): ?>
                <div style="text-align: center; margin-top: 20px;">
                    <a href="mark_sheet_search.php" style="color: #3498db; text-decoration: none; font-weight: bold;">অনুসন্ধান পৃষ্ঠায় ফিরে যান</a>
                </div>
            <?php endif; ?>
        <?php elseif ($student_data): ?>

            <div class="header">
                <h1>মিফতাহুন নাজাত মহিলা মাদরাসা</h1>
                <h2>নেয়ামতপুর,সদর,নোয়াখালী।</h2>
                <h2><?php echo htmlspecialchars($exam_name); ?> পরীক্ষার মার্কশিট</h2>
            </div>

            <div class="student-info">
                <div>
                    <strong>শিক্ষার্থীর নাম:</strong> <?php echo htmlspecialchars($student_data['full_name']); ?>
                </div>
                <div>
                    <strong>শ্রেণি:</strong> <?php echo htmlspecialchars($student_data['class_name']); ?>
                </div>
                <div>
                    <strong>রোল নং:</strong> <?php echo htmlspecialchars($student_data['class_roll']); ?>
                </div>
                <div>
                    <strong>শিক্ষার্থী আইডি:</strong> <?php echo htmlspecialchars($student_data['student_id']); ?>
                </div>
            </div>

            <table>
                <thead>
                    <tr>
                        <th>ক্রমিক নং</th>
                        <th>বিষয়</th>
                        <th>প্রাপ্ত নম্বর</th>
                        <th>গ্রেড</th>
                    </tr>
                </thead>
                <tbody>
                    <?php $i = 1; foreach ($results as $row): ?>
                        <tr>
                            <td><?php echo $i++; ?></td>
                            <td><?php echo htmlspecialchars($row['subject_name']); ?></td>
                            <td><?php echo htmlspecialchars($row['marks_obtained']); ?></td>
                            <td><?php echo htmlspecialchars(calculate_grade($row['marks_obtained'])); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <div class="summary">
                মোট প্রাপ্ত নম্বর: <?php echo $total_marks; ?>
            </div>
            
            <div class="comments-and-signature">
                <div class="comment-box">
                    <strong>শিক্ষক মন্তব্য:</strong>
                    <div style="height: 50px;"></div> </div>

                <div class="comment-box">
                    <strong>অভিভাবক মন্তব্য:</strong>
                    <div style="height: 50px;"></div> </div>
                
                <div class="signature-box">
                    <span class="signature-line">
                        স্বাক্ষর (প্রধান শিক্ষক)
                    </span>
                </div>
            </div>

        <?php endif; ?>

    </div>

</div>

</body>
</html>