<?php
session_start();

// --- বাংলা তারিখ রূপান্তর ফাংশন ---
if (!function_exists('get_bangla_date')) {
    function get_bangla_date($date_str, $format = 'd-m-Y (l)') {
        if (empty($date_str)) {
            return 'N/A';
        }

        $date = new DateTime($date_str);
        $date_format = $date->format($format);

        // বাংলা সংখ্যা
        $bn = ['১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯', '০'];
        $en = ['1', '2', '3', '4', '5', '6', '7', '8', '9', '0'];
        $date_format = str_replace($en, $bn, $date_format);

        // বাংলা বার
        $days_en = ['Saturday', 'Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday'];
        $days_bn = ['শনিবার', 'রবিবার', 'সোমবার', 'মঙ্গলবার', 'বুধবার', 'বৃহস্পতিবার', 'শুক্রবার'];
        $date_format = str_replace($days_en, $days_bn, $date_format);

        return $date_format;
    }
}
// --- ফাইল ইনক্লুড করার জন্য সবচেয়ে নির্ভরযোগ্য রুট পাথ লজিক ---
$ds = DIRECTORY_SEPARATOR;
$project_root_raw = dirname(dirname(__DIR__)); 
$project_root_1 = rtrim($project_root_raw, '/\\'); 

// তিনটি সম্ভাব্য পাথ কনফিগার করা হলো ( db_connection.php, functions.php )
$db_paths = [
    $project_root_1 . $ds . 'includes' . $ds . 'db_connection.php',
    $project_root_1 . $ds . 'includes' . $ds . 'db_connection', // এক্সটেনশন ছাড়া চেষ্টা
];
$functions_paths = [
    $project_root_1 . $ds . 'includes' . $ds . 'functions.php',
];

$found = false;
$db_path = null;
$functions_path = null;

// db_connection ফাইল খুঁজে বের করার চেষ্টা
foreach ($db_paths as $path) {
    if (file_exists($path)) {
        $db_path = $path;
        break;
    }
}

// functions.php ফাইল খুঁজে বের করার চেষ্টা
foreach ($functions_paths as $path) {
    if (file_exists($path)) {
        $functions_path = $path;
        break;
    }
}

// ৩. ফাইল ইনক্লুড করা
if ($db_path && $functions_path) {
    include $db_path;
    include $functions_path;
    $found = true;
} 

// যদি প্রথম চেষ্টা ব্যর্থ হয়, তবে realpath ব্যবহার করে আপেক্ষিক পাথ খোঁজা
if (!$found) {
    $relative_db_path = '..' . $ds . '..' . $ds . '..' . $ds . 'includes' . $ds . 'db_connection.php';
    $relative_functions_path = '..' . $ds . '..' . '..' . $ds . '..' . $ds . 'includes' . $ds . 'functions.php';
    
    $real_db_path = realpath($relative_db_path);
    $real_functions_path = realpath($relative_functions_path);

    if ($real_db_path) {
        include $real_db_path;
        if ($real_functions_path) {
            include $real_functions_path;
        }
        $found = true;
    }
}


if (!$found || !isset($conn)) {
    die("FATAL ERROR: Required core files (db_connection or functions) not found or DB connection failed.");
}

// লগইন ও অনুমতি চেক
if (!function_exists('is_logged_in') || !is_logged_in() || (isset($_SESSION['role']) && $_SESSION['role'] !== 'Admin')) {
    if (function_exists('redirect_to')) {
        redirect_to('../../index.php');
    } else {
        header("Location: ../../index.php");
        exit();
    }
}

// --- ডেটা গ্রহণ ও ভ্যালিডেশন ---
$exam_id = $_GET['exam_id'] ?? null;
$class_id_filter = $_GET['class_id'] ?? null; 

if (empty($exam_id) || !is_numeric($exam_id)) {
    die("ERROR: Invalid Exam ID provided.");
}

// --- ১. রুটিন ডেটা আনার জন্য ফাংশন (DB স্ট্রাকচার: routines, subjects) ---
function get_routine($conn, $exam_id, $class_id) {
    // routines টেবিল subject_id ব্যবহার করে, তাই subjects টেবিলের সাথে JOIN করতে হবে।
    $routine_sql = "
        SELECT 
            s.subject_name, r.exam_date, r.start_time, r.end_time, r.session_type 
        FROM routines r
        JOIN subjects s ON r.subject_id = s.subject_id
        WHERE r.exam_id = ? AND r.class_id = ?
        ORDER BY r.exam_date ASC, r.start_time ASC
    ";
    
    $stmt_routine = $conn->prepare($routine_sql);
    if ($stmt_routine === false) {
        // যদি SQL এ ভুল হয়
        return ["error" => $conn->error]; 
    }

    $stmt_routine->bind_param("ii", $exam_id, $class_id);
    $stmt_routine->execute();
    $result = $stmt_routine->get_result();
    
    $routine_data = [];
    while ($row = $result->fetch_assoc()) {
        $routine_data[] = $row;
    }
    
    $stmt_routine->close();
    return $routine_data;
}

// --- ২. শিক্ষার্থী ও পরীক্ষার তথ্য আনার জন্য SQL ক্যোয়ারি ---
$sql_students = "
    SELECT 
        s.student_id, 
        s.class_id, /* রুটিন আনার জন্য class_id */
        s.first_name,    
        s.last_name,      
        s.class_roll AS roll_no,      
        s.father_name,          
        s.mother_name,          
        c.class_name
    FROM students s
    JOIN classes c ON s.class_id = c.class_id
    WHERE 1=1 
";

$types = ""; 
$params = [];

// ঐচ্ছিক ক্লাস ফিল্টার
if (!empty($class_id_filter) && is_numeric($class_id_filter)) {
    $sql_students .= " AND s.class_id = ?";
    $params[] = $class_id_filter;
    $types .= "i";
}

$sql_students .= " ORDER BY c.class_id, s.class_roll ASC"; 

// পরীক্ষার নাম আলাদাভাবে আনা
$stmt_exam_name = $conn->prepare("SELECT exam_name FROM exams WHERE exam_id = ?");
$stmt_exam_name->bind_param("i", $exam_id);
$stmt_exam_name->execute();
$exam_result = $stmt_exam_name->get_result();
$exam_name = $exam_result->fetch_assoc()['exam_name'] ?? 'পরীক্ষা';
$stmt_exam_name->close();


$stmt_students = $conn->prepare($sql_students); 

if ($stmt_students === false) {
    die("SQL PREPARE ERROR (Students): " . $conn->error);
}

// প্যারামিটার বাইন্ড করা
if (!empty($types)) {
    $stmt_students->bind_param($types, ...$params);
}

// ক্যোয়ারি এক্সিকিউট করা
$stmt_students->execute();
$students_result = $stmt_students->get_result();
// Statement ক্লোজ করা যেতে পারে, কিন্তু নিচে রুটিন আনার জন্য ডেটা পয়েন্টার রিসেট করতে হবে।


// --- ৩. রুটিন ক্যাশিং লজিক (সব কাজ শুরু করার আগে রুটিন ডেটা এনে রাখা) ---
$routines_cache = [];
$unique_class_ids = [];

// সমস্ত ক্লাস আইডিগুলি সংগ্রহ করা
if ($students_result->num_rows > 0) {
    // রেজাল্ট সেটের পয়েন্ট শুরুতে ফিরিয়ে আনা হলো
    $students_result->data_seek(0); 
    while($student_row = $students_result->fetch_assoc()) {
        if (!in_array($student_row['class_id'], $unique_class_ids)) {
            $unique_class_ids[] = $student_row['class_id'];
        }
    }
    // আবার রেজাল্ট সেটের পয়েন্ট শুরুতে ফিরিয়ে আনা হলো, যাতে নিচের লুপে ব্যবহার করা যায়
    $students_result->data_seek(0); 
}

// প্রতিটি ক্লাসের জন্য রুটিন আনা এবং ক্যাশ করা
foreach ($unique_class_ids as $class_id) {
    $routines_cache[$class_id] = get_routine($conn, $exam_id, $class_id);
}


// **গুরুত্বপূর্ণ:** সব DB কাজ শেষ, এখন সংযোগ বন্ধ করা হলো।
if (isset($conn)) {
    $conn->close();
}
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($exam_name ?: 'প্রবেশপত্র'); ?> - প্রিন্ট প্রিভিউ</title>
    <link rel="stylesheet" href="../../css/style.css"> 
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        /* --- সাধারণ স্ক্রীন স্টাইল (প্রিন্ট ছাড়া) --- */
        body {
            background-color: #ecf0f1;
            padding: 30px;
        }
        .print-area {
            display: flex;
            flex-direction: column; 
            flex-wrap: wrap;
            gap: 20px; 
            justify-content: center;
            align-items: center; 
            margin-top: 20px;
        }
        .admit-card-wrapper {
            /* স্ক্রীন সাইজ: A4 পোর্ট্রেট এর আনুমানিক কনটেন্ট চওড়া */
            width: 750px; 
            padding: 10px;
            margin-bottom: 20px;
            border: 1px solid #ddd;
            border-radius: 12px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            background-color: #fff;
            page-break-inside: avoid;
        }

        .admit-card {
            width: 100%;
            height: 280px; 
            padding: 20px;
            border: 1px solid #c0d3d9;
            border-radius: 12px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            background: linear-gradient(135deg, #fefefe, #f9f9f9); 
            font-family: 'Inter', Arial, sans-serif; 
            position: relative;
            overflow: hidden;
            margin-bottom: 15px; 
        }
        .admit-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 10px;
            height: 100%;
            background-color: #3498db; 
        }

        .admit-card-header {
            text-align: center;
            border-bottom: 2px solid #3498db; 
            padding-bottom: 10px;
            margin-bottom: 15px;
        }
        .admit-card-header h3 {
            margin: 0;
            color: #2980b9;
            font-size: 20px; 
            font-weight: 700;
        }
        .admit-card-header p {
            margin: 2px 0 0;
            font-size: 14px;
            color: #555;
            font-style: italic;
        }
        .admit-card-body {
            display: grid;
            grid-template-columns: 3fr 1fr; 
            gap: 15px;
            font-size: 14px; 
        }
        .info-section p {
            margin: 4px 0;
            line-height: 1.3;
            color: #333;
        }
        .info-section strong {
            display: inline-block;
            width: 110px; 
            color: #1c1c1c;
            font-weight: 600;
        }
        /* আসন নং স্টাইল */
        .seat-no-line {
            display: block;
            margin-top: 8px;
            font-size: 14px;
        }
        .seat-no-line strong {
            width: 110px;
        }
        .seat-no-line span {
            font-weight: normal;
            border-bottom: 1px solid #000;
            padding: 0 20px;
            display: inline-block;
            width: 150px; /* লেখার জন্য একটু বেশি জায়গা */
            text-align: center;
        }
        /* আসন নং স্টাইল শেষ */

        .photo-section {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: space-between;
            padding-left: 10px; 
        }
        .student-photo {
            width: 80px; 
            height: 90px;
            border: 2px solid #3498db; 
            background-color: #ecf0f1;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 10px;
            color: #777;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
            margin-bottom: 8px;
        }
        .signature {
            font-size: 11px; 
            text-align: center;
            border-top: 1px dashed #7f8c8d;
            padding-top: 4px;
            width: 100%;
            color: #444;
        }
        /* --- রুটিন ডিজাইন আপডেট: কম্প্যাক্ট এবং স্পষ্ট --- */
        .routine-section {
            margin-top: 12px; 
            padding-top: 8px; 
            border-top: 1px dashed #777; 
        }
        .routine-section h4 {
            text-align: center;
            color: #2c3e50;
            font-size: 14px; 
            margin: 4px 0 8px 0;
            font-weight: 700;
            background-color: #f0f0f0; 
            padding: 5px;
            border-radius: 4px;
            border-bottom: 2px solid #34495e;
        }
        .routine-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 11px; 
            border: 2px solid #34495e; /* টেবিলের চারপাশে গাঢ় বর্ডার */
            margin-bottom: 5px;
        }
        .routine-table th, .routine-table td {
            border: 1px solid #34495e !important; /* কলামের দাগ গাঢ় */
            padding: 4px; 
            text-align: center;
        }
        .routine-table th {
            background-color: #34495e; 
            color: #fff;
            font-weight: 700;
        }
        .routine-table tr:nth-child(even) {
            background-color: #f7f7f7; /* একটি পর একটি সারিকে হালকা ব্যাকগ্রাউন্ড */
        }
        .no-routine {
            text-align: center;
            font-size: 11px;
            color: #e74c3c;
            padding: 6px;
            border: 1px dashed #e74c3c;
            border-radius: 5px;
        }

/* --- প্রিন্টের জন্য CSS (A4 পোর্ট্রেট, স্মার্ট পেজ ব্রেক সহ) --- */
        @media print {
            body {
                background-color: #fff;
                padding: 0;
                margin: 0;
            }
            .print-button, .back-button {
                display: none;
            }
            .print-area {
                width: 100%;
                display: block; 
                padding: 0;
            }
            
            /* প্রতিটি কার্ড মোড়ক - কাটিং এরিয়া */
            .admit-card-wrapper {
                width: 98%; 
                max-width: 100%;
                box-shadow: none;
                margin: 0.25% auto 0.5% auto; /* উপরে-নিচে একটু গ্যাপ */
                border: 1px dashed #000; 
                padding: 5px; 
                
                /* উচ্চতা এখন কন্টেন্ট অনুযায়ী হবে */
                height: auto; 
                box-sizing: border-box; 
                overflow: visible; /* overflow: hidden সরিয়ে দেয়া হয়েছে */
                
                /* স্মার্ট পেজ ব্রেক লজিক: নিশ্চিত করে যে পুরো কার্ডটি পরবর্তী পৃষ্ঠায় যাবে যদি এটি বর্তমান পৃষ্ঠায় ফিট হওয়ার সম্ভাবনা না থাকে */
                page-break-inside: avoid; 
                page-break-after: auto; /* জোর করে পেজ ব্রেক সরানো হলো */
            }
            
            /* যদি কার্ডের মোড়কের ভেতরের মোট উচ্চতা A4 এর অর্ধেক বা তার কম হয়, তবে পাশাপাশি দুটি কার্ড এক পেজে ফিট হবে। CSS নিজে থেকেই সেটা অ্যাডজাস্ট করবে। */
            
            /* যদি দুটি কার্ড একসাথে এক পেজে ধরে, তবে সেটা হবে। যদি একটি কার্ডের রুটিন বড় হওয়ার কারণে অন্যটি জায়গা না পায়, তবে CSS ঐ কার্ডটিকে পরের পেজে নিয়ে যাবে। */
            
            /* নির্দিষ্ট `:nth-child(2n)` লজিকটি সরানো হলো */
            
            .admit-card {
                width: 100%;
                margin: 0;
                border: 1px solid #000; 
                box-shadow: none;
                background: #fff;
                height: auto; /* উচ্চতা কন্টেন্টের উপর নির্ভরশীল */
                padding: 10px; 
            }
            .admit-card::before {
                display: none; 
            }
            
            /* প্রিন্টে রুটিন টেবিল বর্ডার ফিক্স (আগের মতোই) */
            .routine-table {
                font-size: 10px; 
                border: 1px solid #000; 
            }
            .routine-table th, .routine-table td {
                border: 1px solid #000 !important; 
                padding: 3px; 
            }
            .routine-table th {
                background-color: #ccc !important; 
                color: #000 !important;
                -webkit-print-color-adjust: exact; 
                print-color-adjust: exact;
            }
            .routine-section h4 {
                background-color: #e0e0e0 !important;
                color: #000 !important;
                border-bottom: 2px solid #000;
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }
            .routine-table tr:nth-child(even) {
                background-color: #f0f0f0 !important;
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }

            @page {
                size: A4 portrait; 
                margin: 0.1cm; 
            }
        }
    </style>
</head>
<body>

    <a href="admit_card_generator.php" class="back-button" style="margin-left: 20px;"><i class="fas fa-arrow-left"></i> প্রবেশপত্র জেনারেটরে ফিরে যান</a>
    <button class="print-button" onclick="window.print()"><i class="fas fa-print"></i> প্রবেশপত্র প্রিন্ট করুন</button>

    <div class="print-area">
    <?php
    if ($students_result->num_rows > 0) {
        $card_count = 0; 
        while($student = $students_result->fetch_assoc()) {
            $card_count++;
            // student data
            $student_full_name = htmlspecialchars($student['first_name'] . ' ' . $student['last_name']);
            $current_class_id = $student['class_id'];
            
            // ক্যাশ থেকে রুটিন ডেটা আনা
            $routine_data = $routines_cache[$current_class_id] ?? [];
            ?>
            <div class="admit-card-wrapper">
                <div class="admit-card">
                    <div class="admit-card-header">
                        <h3>মিফতাহুন নাজাত মহিলা মাদরাসা</h3>
                        <p>প্রবেশপত্র - <?php echo htmlspecialchars($exam_name); ?></p>
                    </div>
                    <div class="admit-card-body">
                        <div class="info-section">
                            <p><strong>শিক্ষার্থী আইডি:</strong> <?php echo htmlspecialchars($student['student_id']); ?></p>
                            <p><strong>শিক্ষার্থীর নাম:</strong> <?php echo $student_full_name; ?></p>
                            <p><strong>শ্রেণি:</strong> <?php echo htmlspecialchars($student['class_name']); ?></p>
                            <p><strong>রোল নং:</strong> <?php echo htmlspecialchars($student['roll_no']); ?></p>
                            <p><strong>পিতার নাম:</strong> <?php echo htmlspecialchars($student['father_name'] ?? 'N/A'); ?></p>
                            <p><strong>মাতার নাম:</strong> <?php echo htmlspecialchars($student['mother_name'] ?? 'N/A'); ?></p>
                            <p><strong>পরীক্ষা কেন্দ্র:</strong> [পরীক্ষা কেন্দ্রের নাম]</p>
                            <p class="seat-no-line">
                                <strong>আসন নং:</strong> <span>__________</span>
                            </p>
                        </div>
                        <div class="photo-section">
                            <div class="student-photo">
                                ছবি
                            </div>
                            <div class="signature">
                                পরীক্ষা নিয়ন্ত্রকের স্বাক্ষর
                            </div>
                        </div>
                    </div>
                </div>
                <div class="routine-section">
                    <h4><i class="fas fa-calendar-alt"></i> <?php echo htmlspecialchars($student['class_name']); ?> এর পরীক্ষার রুটিন</h4>
                    <?php if (!empty($routine_data) && !isset($routine_data['error'])): ?>
                    <table class="routine-table">
                        <thead>
                            <tr>
                                <th>তারিখ</th>
                                <th>বার</th>
                                <th>সময় (শুরু-শেষ)</th>
                                <th>বিষয়</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            foreach ($routine_data as $routine_row): 
                                // সেশন টাইপ অনুযায়ী 'সকাল' বা 'বিকাল' নির্ধারণ
                                $session_text = '';
                                if (!empty($routine_row['session_type'])) {
                                    $session_text = (strtolower($routine_row['session_type']) == 'morning') ? 'সকাল' : 'দিবা';
                                }

                                // বাংলা সময়
                                $start_time_bn = get_bangla_date($routine_row['start_time'], 'h:i');
                                $end_time_bn = get_bangla_date($routine_row['end_time'], 'h:i');
                            ?>
                            <tr>
                                <td><?php echo get_bangla_date($routine_row['exam_date'], 'd-m-Y'); ?></td>
                                <td><?php echo get_bangla_date($routine_row['exam_date'], 'l'); ?></td>
                                <td><?php echo $start_time_bn . ' - ' . $end_time_bn . ' (' . $session_text . ')'; ?></td>
                                <td><?php echo htmlspecialchars($routine_row['subject_name']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php elseif (isset($routine_data['error'])): ?>
                    <div class="no-routine">রুটিন SQL ত্রুটি: <?php echo htmlspecialchars($routine_data['error']); ?></div>
                    <?php else: ?>
                    <div class="no-routine">এই শ্রেণির জন্য কোনো রুটিন পাওয়া যায়নি।</div>
                    <?php endif; ?>
                </div>
            </div>
            <?php
        }
    } else {
        echo "<h2 style='text-align:center; width:100%; color:#c0392b;'>কোনো শিক্ষার্থী পাওয়া যায়নি!</h2>";
    }
    ?>
    </div>
</body>
</html>