<?php
session_start();
// ডেটাবেস সংযোগ পাথের সঠিকতা নিশ্চিত করুন
include '../../includes/db_connection.php'; 

// নিরাপত্তা যাচাই
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'Admin') {
    die("আপনার এই পাতা ব্যবহারের অনুমতি নেই।");
}

date_default_timezone_set('Asia/Dhaka');

// --- রিপোর্ট মাস নির্বাচন ও ডিফল্ট মান সেট করা ---
$target_month = isset($_GET['target_month']) ? $_GET['target_month'] : 'all';

// SQL-এ ব্যবহারের জন্য ফিল্টার তৈরি
$sql_month_filter = '1'; 
$sql_other_income_filter = '1';
$sql_expense_filter = '1';
$report_title_suffix = '(সর্বকালীন)';

if ($target_month !== 'all' && $target_month !== '') {
    // নির্দিষ্ট মাস ফিল্টার
    $sql_month_filter = "DATE_FORMAT(f.payment_date, '%Y-%m') = '{$target_month}'";
    $sql_other_income_filter = "DATE_FORMAT(oi.income_date, '%Y-%m') = '{$target_month}'";
    $sql_expense_filter = "DATE_FORMAT(e.expense_date, '%Y-%m') = '{$target_month}'";
    $report_title_suffix = '(' . get_bangla_month_name($target_month) . ' মাস)';
}


// বাংলা সংখ্যায় রূপান্তর ফাংশন
function format_to_bangla_number($number) {
    $formatted_number = number_format($number, 0, '.', ','); 
    $english_digits = range(0, 9);
    $bangla_digits = array('০', '১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯');
    return str_replace($english_digits, $bangla_digits, $formatted_number);
}

// বাংলা মাস ও বছর ফাংশন
function get_bangla_month_name($month_year) {
    $timestamp = strtotime($month_year . '-01');
    $month_name = date('F', $timestamp);
    $year = date('Y', $timestamp);
    
    $bangla_months = array(
        'January' => 'জানুয়ারি', 'February' => 'ফেব্রুয়ারি', 'March' => 'মার্চ',
        'April' => 'এপ্রিল', 'May' => 'মে', 'June' => 'জুন',
        'July' => 'জুলাই', 'August' => 'আগস্ট', 'September' => 'সেপ্টেম্বর',
        'October' => 'অক্টোবর', 'November' => 'নভেম্বর', 'December' => 'ডিসেম্বর'
    );
    
    $bangla_year = str_replace(range(0, 9), array('০', '১', '২', '৩', '৪', '৫', '৬', '৭', '৮', '৯'), $year);
    return $bangla_months[$month_name] . ' ' . $bangla_year;
}


// ===========================================
// ১. আয়ের খাত লোড করা
// ===========================================
// ১.১ ফি ক্যাটেগরি (Fees) লোড করা (সংগ্রহ সহ) - প্রাতিষ্ঠানিক আয়
$fee_categories = [];
$fee_sql = "
    SELECT 
        ic.category_id AS id, 
        ic.category_name AS name, 
        COALESCE(SUM(CASE WHEN {$sql_month_filter} THEN f.amount ELSE 0 END), 0) AS total_collected 
    FROM 
        income_categories ic
    LEFT JOIN 
        fees f ON ic.category_id = f.category_id
    GROUP BY 
        ic.category_id, ic.category_name
    ORDER BY 
        ic.category_id ASC
";
$fee_result = $conn->query($fee_sql);

if ($fee_result) {
    while ($row = $fee_result->fetch_assoc()) {
        $fee_categories[] = $row;
    }
}


// ১.২ অন্যান্য আয়ের উৎস (Other Incomes) লোড করা (সংগ্রহ সহ) - অন্যান্য আয়
$other_sources = [];
$other_sql = "
    SELECT 
        isrc.source_id AS id, 
        isrc.source_name AS name, 
        isrc.is_active, 
        COALESCE(SUM(CASE WHEN {$sql_other_income_filter} THEN oi.amount ELSE 0 END), 0) AS total_collected 
    FROM 
        income_sources isrc
    LEFT JOIN 
        other_incomes oi ON isrc.source_name = oi.income_source
    GROUP BY 
        isrc.source_id, isrc.source_name, isrc.is_active
    ORDER BY 
        isrc.source_name ASC
";
$other_result = $conn->query($other_sql);

if ($other_result) {
    while ($row = $other_result->fetch_assoc()) {
        $other_sources[] = $row;
    }
}

// মোট আয়ের জন্য সব আইটেম একত্রিত করা
$all_income_items_for_total = array_merge($fee_categories, $other_sources);
$total_collected_income = array_sum(array_column($all_income_items_for_total, 'total_collected'));

// ===========================================
// ২. ব্যয়ের খাত লোড করা (ব্যয় সহ)
// ===========================================
$expense_categories = [];
$expense_sql = "
    SELECT 
        ec.category_id AS id, 
        ec.category_name AS name, 
        COALESCE(SUM(CASE WHEN {$sql_expense_filter} THEN e.amount ELSE 0 END), 0) AS total_spent
    FROM 
        expense_categories ec
    LEFT JOIN 
        expenses e ON ec.category_id = e.category_id
    WHERE 
        ec.is_active = TRUE
    GROUP BY 
        ec.category_id, ec.category_name
    ORDER BY 
        ec.category_name ASC
";
$expense_result = $conn->query($expense_sql);

if ($expense_result) {
    while ($row = $expense_result->fetch_assoc()) {
        $expense_categories[] = $row;
    }
}

$total_spent_expense = array_sum(array_column($expense_categories, 'total_spent'));
$net_result = $total_collected_income - $total_spent_expense;


// =========================================================
// ৩. মেনুর জন্য ডেটাবেস থেকে মাসগুলির তালিকা তৈরি করা
// =========================================================
$months_list = [];
$months_query = "
    SELECT DISTINCT DATE_FORMAT(payment_date, '%Y-%m') AS month_year 
    FROM fees
    WHERE payment_date IS NOT NULL
    
    UNION 
    
    SELECT DISTINCT DATE_FORMAT(income_date, '%Y-%m') AS month_year 
    FROM other_incomes
    WHERE income_date IS NOT NULL
    
    UNION 
    
    SELECT DISTINCT DATE_FORMAT(expense_date, '%Y-%m') AS month_year 
    FROM expenses
    WHERE expense_date IS NOT NULL
    
    ORDER BY month_year DESC
";
$months_result = $conn->query($months_query);

if ($months_result) {
    while ($row = $months_result->fetch_assoc()) {
        $months_list[] = $row['month_year'];
    }
}


// =========================================================
// ৪. ডিসপ্লের জন্য আয়ের তালিকা তৈরি করা (হেডার সহ)
// =========================================================
$income_display_list = [];

// 1. প্রাতিষ্ঠানিক আয় Header
$income_display_list[] = ['type' => 'header', 'name' => 'প্রাতিষ্ঠানিক আয়'];

// 2. Fee Categories
foreach ($fee_categories as $item) {
    $income_display_list[] = ['type' => 'item', 'name' => $item['name'], 'amount' => $item['total_collected']];
}

// 3. অন্যান্য আয় Header
$income_display_list[] = ['type' => 'header', 'name' => 'অন্যান্য আয়'];

// 4. Other Sources
foreach ($other_sources as $item) {
    $income_display_list[] = ['type' => 'item', 'name' => $item['name'], 'amount' => $item['total_collected']];
}

$expense_display_list = array_values($expense_categories); 

// আয় এবং ব্যয়ের খাতগুলির মধ্যে row-এর সংখ্যা নির্ধারণ
$max_rows = max(count($income_display_list), count($expense_display_list));

// সমস্ত কাজ শেষ, এখন সংযোগ বন্ধ করা হলো।
$conn->close();

?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>আয়-ব্যয় সারাংশ</title>
    <link rel="stylesheet" href="../../public/css/style.css"> 
    <style>
        .container { max-width: 1000px; margin: 50px auto; padding: 20px; background-color: #fff; border-radius: 8px; box-shadow: 0 0 10px rgba(0, 0, 0, 0.1); }
        h2 { text-align: center; color: #333; margin-bottom: 20px; padding-bottom: 10px; border-bottom: 2px solid #eee; }
        .pnl-table { width: 100%; border-collapse: collapse; margin-bottom: 40px; table-layout: fixed; }
        .pnl-table th, .pnl-table td { border: 1px solid #ddd; padding: 12px 8px; text-align: left; }
        .pnl-table thead th { font-weight: bold; color: white; }
        .income-header { background-color: #4CAF50; }
        .expense-header { background-color: #FF5722; }
        .amount-column { text-align: right; font-weight: bold; background-color: #e8f5e9; }
        .spent-column { text-align: right; font-weight: bold; background-color: #fce4ec; }
        .total-row td { border-top: 3px double #000; font-weight: bold; background-color: #fffde7; font-size: 1.1em; }
        .net-result { padding: 15px 20px; text-align: center; border-top: 2px solid #000; background-color: #e0f2f1; font-weight: bold; font-size: 1.3em; color: #333; }
        .profit { color: #004d40; }
        .loss { color: #b71c1c; }
        .income-section-header { 
            font-weight: bold; 
            background-color: #b2dfdb; 
            text-align: center; 
            font-size: 1.1em;
            color: #004d40;
        }
        
        /* Search Bar Style */
        .search-bar { margin-top: 15px; margin-bottom: 20px; padding: 10px; border: 1px solid #ccc; border-radius: 5px; background-color: #f9f9f9; display: flex; align-items: center; justify-content: space-between;}
        .search-bar label { margin-right: 10px; font-weight: bold; color: #555;}
        .search-bar select { padding: 8px; border: 1px solid #ccc; border-radius: 4px; font-size: 1em; }
        .search-bar button { padding: 8px 15px; background-color: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; }
        .search-bar button:hover { background-color: #0056b3; }
        .month-display { font-size: 1.1em; font-weight: bold; color: #333; }

        /* Action Buttons */
        .action-buttons { 
            text-align: right; 
            margin-bottom: 10px; /* সার্চ বারের উপরে একটু গ্যাপ */
            padding-bottom: 10px;
            border-bottom: 1px dashed #ccc; /* সার্চ বারের সাথে আলাদা করতে */
        }
        .action-buttons button, .action-buttons a {
            padding: 10px 15px;
            margin-left: 10px;
            font-size: 1em;
            cursor: pointer;
            border: none;
            border-radius: 5px;
            text-decoration: none;
            display: inline-block;
            font-weight: bold;
        }
        .print-btn { background-color: #607d8b; color: white; }
        .print-btn:hover { background-color: #455a64; }
        .dashboard-btn { background-color: #007bff; color: white; }
        .dashboard-btn:hover { background-color: #0056b3; }

        /* Print Specific Styles */
        @media print {
            .header, .footer, .search-bar, .action-buttons { display: none; }
            .container { margin: 0; padding: 0; box-shadow: none; max-width: 100%; }
            .pnl-table th, .pnl-table td { border-color: #000; }
            .amount-column, .spent-column, .income-section-header, .net-result { background-color: #fff !important; color: #000 !important; -webkit-print-color-adjust: exact; print-color-adjust: exact; }
            .total-row td { background-color: #eee !important; -webkit-print-color-adjust: exact; print-color-adjust: exact; border-top: 2px solid #000; }
        }
    </style>
</head>
<body>
<div class="header">
    <h1>আয় ও ব্যয়ের সার্বিক সারাংশ</h1>
</div>
<div class="content">
    <div class="container">

        <div class="action-buttons">
            <a href="/school_management/public/dashboard.php" class="dashboard-btn">
                🏠 ড্যাশবোর্ডে ফিরুন
            </a>
            <button class="print-btn" onclick="window.print()">
                🖨️ প্রিন্ট করুন
            </button>
        </div>
        
        <div class="search-bar">
            <form action="" method="GET" style="display: flex; align-items: center;">
                <label for="target_month">রিপোর্টের সময়সীমা:</label>
                <select id="target_month" name="target_month" required>
                    <option value="all" <?php echo $target_month === 'all' ? 'selected' : ''; ?>>সকল মাস (All Time)</option>
                    <?php 
                    // ডেটাবেসে পাওয়া মাসগুলি ড্রপডাউনে যোগ করা
                    foreach ($months_list as $month) {
                        $selected = ($target_month === $month) ? 'selected' : '';
                        echo "<option value=\"{$month}\" {$selected}>" . get_bangla_month_name($month) . "</option>";
                    }
                    ?>
                </select>
                <button type="submit">দেখুন</button>
            </form>
            <div class="month-display">
                রিপোর্ট সময়কাল: **<?php echo $report_title_suffix; ?>**
            </div>
        </div>

        <h2>আয় ও ব্যয় তালিকা <?php echo $report_title_suffix; ?></h2>
        
        <table class="pnl-table">
            <thead>
                <tr>
                    <th colspan="2" class="income-header">আয়ের খাতসমূহ</th>
                    <th colspan="2" class="expense-header">ব্যয়ের খাতসমূহ</th>
                </tr>
                <tr>
                    <th class="income-header" style="width: 40%;">খাত</th>
                    <th class="income-header" style="width: 10%; text-align: right;">মোট সংগ্রহ (৳)</th>
                    
                    <th class="expense-header" style="width: 40%;">খাত</th>
                    <th class="expense-header" style="width: 10%; text-align: right;">মোট ব্যয় (৳)</th>
                </tr>
            </thead>
            <tbody>
                <?php 
                for ($i = 0; $i < $max_rows; $i++): 
                    $income_item = $income_display_list[$i] ?? null;
                    $expense_item = $expense_display_list[$i] ?? null;
                ?>
                <tr>
                    <?php if ($income_item): ?>
                        <?php if ($income_item['type'] === 'header'): ?>
                            <td colspan="2" class="income-section-header"><?php echo htmlspecialchars($income_item['name']); ?></td>
                        <?php else: ?>
                            <td class="item-name"><?php echo htmlspecialchars($income_item['name']); ?></td>
                            <td class="amount-column"><?php echo format_to_bangla_number($income_item['amount']); ?></td>
                        <?php endif; ?>
                    <?php else: ?>
                        <td class="item-name"></td>
                        <td class="amount-column"></td>
                    <?php endif; ?>

                    <?php if ($expense_item): ?>
                        <td class="item-name"><?php echo htmlspecialchars($expense_item['name']); ?></td>
                        <td class="spent-column"><?php echo format_to_bangla_number($expense_item['total_spent']); ?></td>
                    <?php else: ?>
                        <td class="item-name"></td>
                        <td class="spent-column"></td>
                    <?php endif; ?>
                </tr>
                <?php endfor; ?>
                
                <tr class="total-row">
                    <td style="text-align: right;">মোট আয়:</td>
                    <td class="amount-column"><?php echo format_to_bangla_number($total_collected_income); ?></td>
                    <td style="text-align: right;">মোট ব্যয়:</td>
                    <td class="spent-column"><?php echo format_to_bangla_number($total_spent_expense); ?></td>
                </tr>
            </tbody>
        </table>

        <?php 
            $result_text = '';
            $result_class = '';
            if ($net_result >= 0) {
                $result_text = 'নিট লাভ: ' . format_to_bangla_number(abs($net_result));
                $result_class = 'profit';
            } else {
                $result_text = 'নিট ক্ষতি: ' . format_to_bangla_number(abs($net_result));
                $result_class = 'loss';
            }
        ?>
        <div class="net-result">
            ফলাফল: <span class="<?php echo $result_class; ?>"><?php echo $result_text; ?> ৳</span>
        </div>
        
    </div>
</div>
<div class="footer">
    <p>&copy; <?php echo date("Y"); ?> স্কুল ম্যানেজমেন্ট সিস্টেম। সর্বস্বত্ব সংরক্ষিত।</p>
</div>
</body>
</html>